﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Diagnostics;
using VeteransAffairs.Registries.BusinessAHOBPR;
using VeteransAffairs.Registries.BusinessAHOBPR.VeteranStatusService;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR.Emis
{
    public class VeteranStatusManager : AHOBPRBaseBO, IUpdateManager
    {
        const string DEFAULTDEATHINDICATORCODE = "N";
        const string EMPTYDATE = "0001-01-01";
        const string YESCODE = "y";

        string connectionInfo = string.Empty;
        public VeteranStatusManager() { }
        public VeteranStatusManager(string dbConnectionInfo)
        {
            connectionInfo = dbConnectionInfo;
        }

        public string Name { get { return "VeteranStatus"; } }

        public bool Update(string edipi, int registrantId)
        {
            var successfullyProcessed = true;
            VeteranStatus data = null;

            //Get veteranstatus info from emis using edipi
            var importer = new VeteranStatusImporter();
            try
            {
                data = importer.GetVeteranStatus(edipi);
            }
            catch (Exception ex)
            {
                var errormessage = "VeteranStatusImporter.Update(): Exception VeteranStatusImporter.GetVeteranStatus() for registrant with registrant_id: " + registrantId + "  " + ex.Message;
                Trace.WriteLine(errormessage);
                AHOBPRLogger.LogErrorMessage("eMIS Error", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, errormessage);
                successfullyProcessed = false;
            }
            try
            {
                if (data != null)
                {
                    successfullyProcessed = ProcessVeteranStatus(data);
                }
                else
                {
                    var errormessage = "VeteranStatusImporter.Update(): null VeteranStatus object returned when calling VeteranStatusImporter.GetVeteranStatus() for registrant with registrant_id: " + registrantId;
                    Trace.WriteLine(errormessage);
                    AHOBPRLogger.LogErrorMessage("eMIS Error", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, errormessage);
                    successfullyProcessed = false;
                }
                //update registrant object with veteranstatus information
            }
            catch (Exception ex)
            {
                var errormessage = "VeteranStatusImporter.Update(): 2nd Stage Exception VeteranStatusImporter.ProcessVeteranStatus() for registrant with registrant_id: " + registrantId + "  " + ex.Message;
                Trace.WriteLine(errormessage);
                AHOBPRLogger.LogErrorMessage("eMIS Error", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, errormessage);
                successfullyProcessed = false;
            }

            return successfullyProcessed;
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        bool ProcessVeteranStatus(VeteranStatus data)
        {
           bool result = true;
            //if not null, get a registrant from the DB
            
            if (data != null)
            {
                var manager = new AHOBPRRegistrantManager();
                using (var db = GetLocalContext(connectionInfo))
                {
                    db.DeferredLoadingEnabled = false;
                                    
                    var registrant = db.REGISTRANTs.FirstOrDefault(e => e.EDIPI == data.edipi);
                    //Check if Registrant is null, otherwise this is all pointless. 

                    if (registrant != null)
                    {                     
                        UpdateRegistrantObjWithVeteranStatusInfo(data, registrant, db);
                        db.SubmitChanges();
                    }
                    else //The registrant object REALLY SHOULDN'T be null, so add to the log and call shennanigans.
                    {
                        var errormessage = "Database call couldn't find registrant by EDIPI: " + data.edipi;
                        Trace.WriteLine("VeteranStatusImporter.ProcessVeteranStatus: " + errormessage);
                        AHOBPRLogger.LogErrorMessage("DB Error", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, errormessage);
                        result = false;
                    }
                }
            }
            //A returnStatus of -1 means the attempt to save to the DB failed. Anything else is the ammount and attempts to save to the DB from 1 through 20
            return result;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="data"></param>
        /// <param name="registrant"></param>
        /// <returns></returns>
        void UpdateRegistrantObjWithVeteranStatusInfo(VeteranStatus data, REGISTRANT registrant, AHOBPRDataAccess db)
        {
            if (data == null) throw new ArgumentNullException(nameof(data));
            if (registrant == null) throw new ArgumentNullException(nameof(registrant));
            if (db == null) throw new ArgumentNullException(nameof(db));

            Trace.WriteIf(data?.socialSecurityNumber != registrant.Snum, $"Registrant: {registrant.REGISTRANT_ID}'s social security number does not match with eMIS's social security number");
            Trace.WriteIf(data?.personFirstName?.ToLower() != registrant.FIRST_NAME.ToLower(), $"Registrant: {registrant.REGISTRANT_ID}'s first name does not match eMIS's entry.");
            Trace.WriteIf(data?.personLastName?.ToLower() != registrant.LAST_NAME.ToLower(), $"Registrant: {registrant.REGISTRANT_ID}'s last name does not match eMIS's entry.");
            Trace.WriteIf(data?.personBirthDate != registrant.BIRTH_DATE, $"Registrant: {registrant.REGISTRANT_ID}'s birth date does not match eMIS's entry.");
            UpdateMaritalStatus(data, registrant);
            UpdateEducationLevel(data, registrant);
            UpdateRace(data, registrant);
            UpdateEthnicAffinityCode(data, registrant, db);
            UpdateDeathIndicator(data, registrant);
            UpdateDeathDate(data, registrant);
            UpdateCurrentBranchOfService(registrant);
            Update911DeploymentIndicators(data, registrant);
            UpdateServiceStatus(data, registrant);
        }

        private void UpdateCurrentBranchOfService(REGISTRANT registrant)
        {
            var bosManager = new BranchesOfServiceManager();
            var branchesOfService = bosManager.GetRegistrantsBranchesOfService(registrant.EDIPI);
            if (branchesOfService.Keys.Count() > 0)
            {
                var current = branchesOfService.OrderByDescending(bos => bos.Key.Start).First().Value;
                registrant.STD_BRANCH_OF_SERVICE_ID_CURRENT = AHOBPRShared.GetStdBranchOfServiceId(current);
            }
        }

        private void UpdateServiceStatus(VeteranStatus data, REGISTRANT registrant)
        {
            var rssmanager = new RegistrantServiceStatusMapper();
            registrant.STD_SERVICE_STATUS_ID = rssmanager.GetServiceStatus(registrant.EDIPI);
        }
        #region Registrant fields to update 

        private void UpdateDeathDate(VeteranStatus data, REGISTRANT registrant)
        {
            if (data.personDeathDate != DateTime.Parse(EMPTYDATE) && !data.personDeathDate.Equals(registrant.DEATH_DATE))
            {
                registrant.DEATH_DATE = data.personDeathDate;
            }
        }

        private void UpdateDeathIndicator(VeteranStatus input, REGISTRANT registrant)
        {
            var deathIndicatorCode = input?.personDeathIndicatorCode ?? DEFAULTDEATHINDICATORCODE;
            registrant.DECEASED_FLAG = TranslateDeathIndicatorCode(deathIndicatorCode);
        }
        private void UpdateEthnicAffinityCode(VeteranStatus input, REGISTRANT registrant, AHOBPRDataAccess db)
        {
            if (InputNotNullOrIdenticalToExisting(input.ethnicAffinityCode, registrant.ETHNICITY)) {
                registrant.ETHNICITY = input.ethnicAffinityCode;
            }
        }
        private static void UpdateMaritalStatus(VeteranStatus input, REGISTRANT registrant)
        {
            if (InputNotNullOrIdenticalToExisting(input.maritalStatusCode, registrant.MARITAL_STATUS))
            {
                registrant.MARITAL_STATUS = input.maritalStatusCode;
            }
        }
        private static void UpdateEducationLevel(VeteranStatus input, REGISTRANT registrant)
        {
            if (InputNotNullOrIdenticalToExisting(input.educationLevelCode, registrant.EDUCATION_LEVEL))
            {
                registrant.EDUCATION_LEVEL = input.educationLevelCode;
            }
        }

        private static void UpdateRace(VeteranStatus input, REGISTRANT registrant)
        {
            if (InputNotNullOrIdenticalToExisting(input.raceCode, registrant.RACE))
            {
                registrant.RACE = input.raceCode;
            }
        }

        static void Update911DeploymentIndicators(VeteranStatus data, REGISTRANT registrant)
        {
            if (InputNotNullOrIdenticalToExisting(data.pre911DeploymentIndicator, registrant.PRE_911_DEPLOYMENT))
            {
                registrant.PRE_911_DEPLOYMENT = data.pre911DeploymentIndicator;
            }
            if (InputNotNullOrIdenticalToExisting(data.post911DeploymentIndicator, registrant.POST_911_DEPLOYMENT))
            {
                registrant.POST_911_DEPLOYMENT = data.post911DeploymentIndicator;
            }
        }
        #endregion
        #region helper methods
        bool TranslateDeathIndicatorCode(string code)
        {
            return code.ToLower().Equals(YESCODE);
        }
        static bool InputNotNullOrIdenticalToExisting(string dataField, string registrantField)
        {
            var result = false;
            //Don't update a possibly populated registrant field if the data field is empty
            if (!string.IsNullOrEmpty(dataField))
            {
                result = !dataField.Equals(registrantField, StringComparison.OrdinalIgnoreCase);
            }
            return result;
        }
        #endregion  
    }
}
